/*
 * Decompiled with CFR 0.152.
 */
package moe.plushie.armourers_workshop.core.data;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import moe.plushie.armourers_workshop.api.core.IResultHandler;
import moe.plushie.armourers_workshop.core.data.ticket.Ticket;
import moe.plushie.armourers_workshop.core.utils.Executors;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;

public class DataTransformer<K, V, T> {
    private final ExecutorService mainThread;
    private final ExecutorService transformThread;
    private final Optional<ExecutorService> cleanThread;
    private final LoadHandler<K, T> loader;
    private final TransformHandler<K, T, V> transformer;
    private final LinkedList<Entry> loadQueue = new LinkedList();
    private final LinkedList<Entry> transformQueue = new LinkedList();
    private final ConcurrentHashMap<K, Entry> allEntries = new ConcurrentHashMap();
    private final AtomicInteger loading = new AtomicInteger(0);
    private final AtomicInteger transforming = new AtomicInteger(0);
    private final int maxLoadCount;
    private final int maxTransformCount;

    protected DataTransformer(ThreadFactory config, LoadHandler<K, T> loader, TransformHandler<K, T, V> transformer, CleanHandler<K> cleaner, int maxLoadCount, int maxTransformCount) {
        this.mainThread = Executors.newFixedThreadPool(1, config);
        this.transformThread = Executors.newFixedThreadPool(maxTransformCount, config);
        this.cleanThread = this.createCleanThread(cleaner);
        this.loader = loader;
        this.transformer = transformer;
        this.maxLoadCount = maxLoadCount;
        this.maxTransformCount = maxTransformCount;
    }

    public void remove(K key) {
        Entry entry = this.allEntries.remove(key);
        if (entry != null && entry.isPending()) {
            entry.abort(new RuntimeException("user cancelled!"));
        }
    }

    @Nullable
    public Pair<V, Exception> get(K key) {
        Entry entry = this.getEntry(key);
        if (entry != null) {
            return entry.transformedData;
        }
        return null;
    }

    @Nullable
    public Pair<V, Exception> getOrLoad(Ticket<K> ticket) {
        Entry entry = this.getEntryAndCreate(ticket.get());
        if (!entry.isCompleted()) {
            this.load(ticket, null);
        } else {
            entry.updateTicket(ticket);
        }
        return entry.transformedData;
    }

    public void load(Ticket<K> ticket, IResultHandler<V> resultHandler) {
        Entry entry = this.getEntryAndCreate(ticket.get());
        entry.updateTicket(ticket);
        entry.listen(resultHandler);
        if (entry.isCompleted()) {
            return;
        }
        entry.elevate(ticket.priority());
        this.mainThread.execute(() -> {
            if (!entry.isLoading) {
                entry.isLoading = true;
                this.loadQueue.add(entry);
            }
            this.dispatchIfNeeded();
        });
    }

    public void clear() {
        this.loadQueue.clear();
        this.transformQueue.clear();
        this.allEntries.clear();
        this.loading.set(0);
        this.transforming.set(0);
    }

    public void shutdown() {
        this.mainThread.shutdown();
        this.transformThread.shutdown();
        this.cleanThread.ifPresent(ExecutorService::shutdown);
        this.clear();
    }

    private void dispatchIfNeeded() {
        this.doLoadIfNeeded();
        this.doTransformIfNeeded();
    }

    private void doLoadIfNeeded() {
        Entry entry;
        if (this.loading.get() >= this.maxLoadCount) {
            return;
        }
        while ((entry = this.getLastTask(this.loadQueue)) != null) {
            if (entry.isCompleted()) continue;
            this.load(entry);
            break;
        }
    }

    private void doTransformIfNeeded() {
        Entry entry;
        if (this.transforming.get() >= this.maxTransformCount) {
            return;
        }
        while ((entry = this.getLastTask(this.transformQueue)) != null) {
            Exception error;
            if (entry.isCompleted() || (error = entry.getLoadedError()) != null) continue;
            Object value = entry.getLoadedValue();
            this.transform(value, entry);
            break;
        }
    }

    private void doCleanIfNeeded(Duration duration, Consumer<K> handler) {
        long startTime = System.currentTimeMillis() - duration.toMillis();
        LinkedList cleanQueue = new LinkedList();
        this.allEntries.forEach((key, entry) -> {
            if (entry.expiredTime < 0L) {
                if (!entry.checkTicket()) {
                    entry.expiredTime = System.currentTimeMillis();
                }
            } else if (entry.expiredTime < startTime) {
                cleanQueue.add(entry.key);
            }
        });
        cleanQueue.forEach(it -> {
            this.allEntries.remove(it);
            handler.accept(it);
        });
    }

    private void load(Entry entry) {
        this.loading.incrementAndGet();
        this.loader.accept(entry.key, (result, exception) -> this.mainThread.execute(() -> {
            this.loading.decrementAndGet();
            entry.receiveLoadResult(result, exception);
            if (!entry.isTransforming) {
                entry.isTransforming = true;
                this.transformQueue.add(entry);
            }
            this.dispatchIfNeeded();
        }));
    }

    private void transform(T value, Entry entry) {
        this.transforming.incrementAndGet();
        this.transformThread.execute(() -> this.transformer.accept(entry.key, value, (result, exception) -> this.mainThread.execute(() -> {
            this.transforming.decrementAndGet();
            entry.receiveTransformResult(result, exception);
            this.dispatchIfNeeded();
        })));
    }

    private Entry getEntry(K key) {
        return this.allEntries.get(key);
    }

    private Entry getEntryAndCreate(K key) {
        return this.allEntries.computeIfAbsent(key, x$0 -> new Entry(this, x$0));
    }

    @Nullable
    private Entry getLastTask(List<Entry> queue) {
        if (queue.isEmpty()) {
            return null;
        }
        Entry lastEntry = null;
        for (Entry entry : queue) {
            if (lastEntry != null && !(entry.priority > lastEntry.priority)) continue;
            lastEntry = entry;
        }
        queue.remove(lastEntry);
        return lastEntry;
    }

    private Optional<ExecutorService> createCleanThread(CleanHandler<K> cleanHandler) {
        if (cleanHandler == null) {
            return Optional.empty();
        }
        ScheduledExecutorService scheduledExecutor = Executors.newSingleThreadScheduledExecutor();
        Duration duration = cleanHandler.duration;
        Consumer handler = cleanHandler.handler;
        long processRate = Math.max(duration.toMillis() / 2L, 500L);
        scheduledExecutor.scheduleAtFixedRate(() -> this.doCleanIfNeeded(duration, handler), 0L, processRate, TimeUnit.MILLISECONDS);
        return Optional.of(scheduledExecutor);
    }

    protected static class CleanHandler<K> {
        private final Duration duration;
        private final Consumer<K> handler;

        public CleanHandler(Duration duration, Consumer<K> handler) {
            this.duration = duration;
            this.handler = handler;
        }
    }

    public static interface LoadHandler<T1, T2> {
        public void accept(T1 var1, IResultHandler<T2> var2);
    }

    public static interface TransformHandler<T1, T2, T3> {
        public void accept(T1 var1, T2 var2, IResultHandler<T3> var3);
    }

    private class Entry {
        private final K key;
        private final HashSet<Ticket<K>> tickets;
        private ArrayList<IResultHandler<V>> callbacks;
        private Pair<T, Exception> loadedData;
        private Pair<V, Exception> transformedData;
        private float priority = 0.0f;
        private long expiredTime = 0L;
        private boolean isLoading = false;
        private boolean isTransforming = false;

        Entry(DataTransformer dataTransformer, K key) {
            this.key = key;
            this.tickets = new HashSet();
        }

        public synchronized void updateTicket(Ticket<K> ticket) {
            this.expiredTime = -1L;
            this.tickets.add(ticket);
        }

        public synchronized boolean checkTicket() {
            this.tickets.removeIf(Ticket::invalid);
            return !this.tickets.isEmpty();
        }

        public void elevate(float priority) {
            if (this.priority < priority) {
                this.priority = priority;
            }
        }

        public void listen(IResultHandler<V> callback) {
            if (callback == null) {
                return;
            }
            if (this.transformedData != null) {
                callback.apply(this.transformedData.getKey(), (Exception)this.transformedData.getValue());
                return;
            }
            if (this.callbacks == null) {
                this.callbacks = new ArrayList();
            }
            this.callbacks.add(callback);
        }

        public void receiveLoadResult(T value, Exception exception) {
            this.loadedData = Pair.of(value, (Object)exception);
            this.isLoading = false;
        }

        public void receiveTransformResult(V value, Exception exception) {
            this.transformedData = Pair.of(value, (Object)exception);
            this.isTransforming = false;
            this.sendNotify();
        }

        public void abort(Exception exception) {
            this.transformedData = Pair.of(null, (Object)exception);
            this.sendNotify();
        }

        public void sendNotify() {
            ArrayList callbacks = this.callbacks;
            this.callbacks = null;
            if (callbacks == null || callbacks.isEmpty()) {
                return;
            }
            for (IResultHandler callback : callbacks) {
                callback.apply(this.transformedData.getKey(), (Exception)this.transformedData.getValue());
            }
        }

        @Nullable
        public T getLoadedValue() {
            if (this.loadedData != null) {
                return this.loadedData.getKey();
            }
            return null;
        }

        @Nullable
        public Exception getLoadedError() {
            if (this.loadedData != null) {
                return (Exception)this.loadedData.getValue();
            }
            return null;
        }

        public boolean isPending() {
            return this.isLoading || this.isTransforming;
        }

        public boolean isCompleted() {
            return this.transformedData != null;
        }
    }

    public static class Builder<K, V, T> {
        private int maxLoadCount = 4;
        private int maxTransformCount = 4;
        private ThreadFactory configure;
        private LoadHandler<K, T> loader;
        private TransformHandler<K, T, V> transformer;
        private CleanHandler<K> cleaner;

        public Builder<K, V, T> thread(String name, int newPriority) {
            this.configure = r -> {
                Thread thread = new Thread(r, name);
                thread.setPriority(newPriority);
                return thread;
            };
            return this;
        }

        public Builder<K, V, T> loadCount(int count) {
            this.maxLoadCount = count;
            return this;
        }

        public Builder<K, V, T> transformCount(int count) {
            this.maxTransformCount = count;
            return this;
        }

        public Builder<K, V, T> loader(LoadHandler<K, T> handler) {
            this.loader = handler;
            return this;
        }

        public Builder<K, V, T> transformer(TransformHandler<K, T, V> handler) {
            this.transformer = handler;
            return this;
        }

        public Builder<K, V, T> cleaner(Duration duration, Consumer<K> handler) {
            this.cleaner = new CleanHandler<K>(duration, handler);
            return this;
        }

        public DataTransformer<K, V, T> build() {
            return new DataTransformer<K, V, T>(this.configure, this.loader, this.transformer, this.cleaner, this.maxLoadCount, this.maxTransformCount);
        }
    }
}

