/*
 * Decompiled with CFR 0.152.
 */
package dev.lucaargolo.charta.editor;

import dev.lucaargolo.charta.editor.ToolPanel;
import dev.lucaargolo.charta.utils.CardImage;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.datatransfer.DataFlavor;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.prefs.Preferences;
import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.TransferHandler;
import javax.swing.UIManager;

public class CardEditor
extends JFrame {
    private static final Logger LOGGER = Logger.getLogger("CardEditor");
    private static final Preferences PREFERENCES = Preferences.userNodeForPackage(CardEditor.class);
    private static final String LAST_DIRECTORY = "lastDirectory";
    private static final int CARD_SCALE = 10;
    private static final int SCALED_WIDTH = 250;
    private static final int SCALED_HEIGHT = 350;
    private final Deque<CardImage> undoHistory = new ArrayDeque<CardImage>();
    private final Deque<CardImage> redoHistory = new ArrayDeque<CardImage>();
    private CardImage currentImage;
    private final JLabel imageLabel;
    private final ToolPanel toolPanel;
    private int draggingButton = 0;

    public CardEditor() {
        this.setTitle("Card Editor");
        this.setSize(800, 600);
        this.setDefaultCloseOperation(3);
        this.setLayout(new BorderLayout());
        this.setTransferHandler(new TransferHandler(){

            @Override
            public boolean canImport(TransferHandler.TransferSupport support) {
                return support.isDataFlavorSupported(DataFlavor.javaFileListFlavor);
            }

            @Override
            public boolean importData(TransferHandler.TransferSupport support) {
                if (!this.canImport(support)) {
                    return false;
                }
                try {
                    List files = (List)support.getTransferable().getTransferData(DataFlavor.javaFileListFlavor);
                    if (!files.isEmpty()) {
                        CardEditor.this.loadImageFile((File)files.getFirst());
                    }
                    return true;
                }
                catch (Exception e) {
                    return false;
                }
            }
        });
        this.imageLabel = new JLabel();
        this.imageLabel.setHorizontalAlignment(0);
        this.add((Component)this.imageLabel, "Center");
        JMenuBar menuBar = new JMenuBar();
        JMenu fileMenu = new JMenu("File");
        JMenuItem newItem = new JMenuItem("New Image");
        JMenuItem loadItem = new JMenuItem("Load Image");
        JMenuItem saveItem = new JMenuItem("Save Image");
        fileMenu.add(newItem);
        fileMenu.add(loadItem);
        fileMenu.add(saveItem);
        menuBar.add(fileMenu);
        JMenu utilsMenu = new JMenu("Utils");
        JMenuItem convertAtlas = new JMenuItem("Convert Atlas");
        JMenuItem fixColors = new JMenuItem("Fix Colors");
        JMenuItem generatePaletteImage = new JMenuItem("Generate PNG Palette");
        JMenuItem generatePaletteFile = new JMenuItem("Generate GPL Palette");
        utilsMenu.add(convertAtlas);
        utilsMenu.add(fixColors);
        utilsMenu.add(generatePaletteImage);
        utilsMenu.add(generatePaletteFile);
        menuBar.add(utilsMenu);
        this.setJMenuBar(menuBar);
        newItem.addActionListener(e -> this.createNewImage());
        loadItem.addActionListener(e -> this.loadCardImage());
        saveItem.addActionListener(e -> this.saveCardImage());
        convertAtlas.addActionListener(e -> this.convertCardAtlas());
        fixColors.addActionListener(e -> this.fixCardColors());
        generatePaletteImage.addActionListener(e -> this.generatePNGPalette());
        generatePaletteFile.addActionListener(e -> this.generateGPLPalette());
        MouseAdapter mouseAdapter = new MouseAdapter(){

            @Override
            public void mousePressed(MouseEvent e) {
                this.validEvent(e, e.getButton(), true);
                CardEditor.this.draggingButton = e.getButton();
                CardEditor.this.imageLabel.requestFocus();
            }

            @Override
            public void mouseDragged(MouseEvent e) {
                this.validEvent(e, CardEditor.this.draggingButton, false);
                CardEditor.this.imageLabel.requestFocus();
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                this.validEvent(e, CardEditor.this.draggingButton, true);
                CardEditor.this.imageLabel.requestFocus();
            }

            @Override
            public void mouseMoved(MouseEvent e) {
                CardEditor.this.imageLabel.requestFocus();
            }

            private void validEvent(MouseEvent e, int button, boolean save) {
                Dimension dimension = CardEditor.this.imageLabel.getSize();
                int left = dimension.width / 2 - 125;
                int top = dimension.height / 2 - 175;
                int x = (e.getX() - left) / 10;
                int y = (e.getY() - top) / 10;
                CardEditor.this.paintImage(button, x, y, save);
            }
        };
        this.imageLabel.addKeyListener(new KeyAdapter(){

            @Override
            public void keyPressed(KeyEvent e) {
                if (e.isControlDown()) {
                    switch (e.getKeyCode()) {
                        case 90: {
                            CardEditor.this.handleUndo();
                            break;
                        }
                        case 89: {
                            CardEditor.this.handleRedo();
                        }
                    }
                }
            }
        });
        this.imageLabel.setFocusable(true);
        this.imageLabel.addMouseListener(mouseAdapter);
        this.imageLabel.addMouseMotionListener(mouseAdapter);
        this.toolPanel = new ToolPanel();
        this.add((Component)this.toolPanel, "South");
        this.createNewImage();
        this.updateImage();
    }

    private void paintImage(int button, int x, int y, boolean save) {
        if (this.currentImage != null && this.toolPanel != null) {
            if (save) {
                if (!this.undoHistory.isEmpty()) {
                    CardImage last = this.undoHistory.getLast();
                    if (!last.equals(this.currentImage)) {
                        this.handleHistory();
                    }
                } else {
                    this.handleHistory();
                }
            }
            if (this.isValidButton(button) && x >= 0 && x < 25 && y >= 0 && y < 35) {
                int colorIndex = button == 1 ? this.toolPanel.getLeftIndex() : this.toolPanel.getRightIndex();
                int alphaIndex = this.toolPanel.getAlphaIndex();
                if (!this.toolPanel.isFilling() && !this.toolPanel.isErasing()) {
                    this.currentImage.setPixel(x, y, colorIndex, alphaIndex);
                } else if (this.toolPanel.isErasing()) {
                    this.currentImage.setPixel(x, y, 0, 0);
                } else {
                    byte targetPixel = (byte)(alphaIndex << 6 | colorIndex & 0x3F);
                    byte currentPixel = this.currentImage.getPixel(x, y);
                    if (currentPixel != targetPixel) {
                        this.fillArea(x, y, currentPixel, colorIndex, alphaIndex);
                    }
                }
            }
            this.updateImage();
        }
    }

    private void fillArea(int startX, int startY, byte targetPixel, int colorIndex, int alphaIndex) {
        LinkedList<Point> queue = new LinkedList<Point>();
        queue.add(new Point(startX, startY));
        while (!queue.isEmpty()) {
            Point p = (Point)queue.poll();
            int x = p.x;
            int y = p.y;
            if (x < 0 || x >= 25 || y < 0 || y >= 35 || this.currentImage.getPixel(x, y) != targetPixel) continue;
            this.currentImage.setPixel(x, y, colorIndex, alphaIndex);
            queue.add(new Point(x + 1, y));
            queue.add(new Point(x - 1, y));
            queue.add(new Point(x, y + 1));
            queue.add(new Point(x, y - 1));
        }
    }

    private boolean isValidButton(int button) {
        return button == 1 || button == 3;
    }

    private void handleHistory() {
        this.undoHistory.addLast(this.currentImage.copy());
        if (this.undoHistory.size() > 1000) {
            this.undoHistory.removeFirst();
        }
        this.redoHistory.clear();
    }

    private void handleUndo() {
        if (!this.undoHistory.isEmpty()) {
            this.redoHistory.addLast(this.currentImage);
            CardImage lastImage = this.undoHistory.removeLast();
            if (lastImage.equals(this.currentImage)) {
                lastImage = this.undoHistory.removeLast();
            }
            this.currentImage = lastImage;
        }
        this.updateImage();
    }

    private void handleRedo() {
        if (!this.redoHistory.isEmpty()) {
            this.undoHistory.addLast(this.currentImage);
            this.currentImage = this.redoHistory.removeLast();
        }
        this.updateImage();
    }

    private void updateImage() {
        BufferedImage displayImage = new BufferedImage(25, 35, 2);
        for (int x = 0; x < 25; ++x) {
            for (int y = 0; y < 35; ++y) {
                displayImage.setRGB(x, y, this.currentImage.getARGBPixel(x, y));
            }
        }
        this.imageLabel.setIcon(new ImageIcon(displayImage.getScaledInstance(250, 350, 16)));
        this.imageLabel.revalidate();
        this.imageLabel.repaint();
    }

    private void createNewImage() {
        this.currentImage = new CardImage();
        this.undoHistory.clear();
        this.redoHistory.clear();
        for (int x = 0; x < 25; ++x) {
            for (int y = 0; y < 35; ++y) {
                this.currentImage.setPixel(x, y, 7, 3);
            }
        }
        this.updateImage();
    }

    private JFileChooser getFileChooser() {
        JFileChooser fileChooser = new JFileChooser();
        String lastDir = PREFERENCES.get(LAST_DIRECTORY, null);
        if (lastDir != null) {
            fileChooser.setCurrentDirectory(new File(lastDir));
        }
        return fileChooser;
    }

    private void loadCardImage() {
        JFileChooser fileChooser = this.getFileChooser();
        int returnValue = fileChooser.showOpenDialog(this);
        if (returnValue == 0) {
            File selectedFile = fileChooser.getSelectedFile();
            PREFERENCES.put(LAST_DIRECTORY, selectedFile.getParent());
            this.loadImageFile(selectedFile);
        }
    }

    private void loadImageFile(File selectedFile) {
        try {
            if (selectedFile.getName().endsWith(".mccard")) {
                this.currentImage = CardImage.loadFromFile(selectedFile);
                this.undoHistory.clear();
                this.redoHistory.clear();
            } else {
                BufferedImage image = ImageIO.read(selectedFile);
                this.currentImage = new CardImage();
                this.undoHistory.clear();
                this.redoHistory.clear();
                int width = Math.min(image.getWidth(), 25);
                int height = Math.min(image.getHeight(), 35);
                for (int x = 0; x < width; ++x) {
                    for (int y = 0; y < height; ++y) {
                        int argb = image.getRGB(x, y);
                        this.currentImage.setARGBPixel(x, y, argb);
                    }
                }
            }
            this.updateImage();
        }
        catch (IOException e) {
            LOGGER.log(Level.SEVERE, "Error loading image: " + String.valueOf(selectedFile.getAbsoluteFile()), e);
            JOptionPane.showMessageDialog(this, "Error loading image.");
        }
    }

    private void convertCardAtlas() {
        JFileChooser fileChooser = this.getFileChooser();
        int returnValue = fileChooser.showOpenDialog(this);
        if (returnValue == 0) {
            File selectedFile = fileChooser.getSelectedFile();
            PREFERENCES.put(LAST_DIRECTORY, selectedFile.getParent());
            try {
                String fileName = selectedFile.getName();
                String cardName = fileName.contains(".") ? fileName.substring(0, fileName.lastIndexOf(46)) : fileName;
                File outputFile = new File(String.valueOf(selectedFile.toPath().getParent()) + File.separator + cardName);
                CardImage.saveCards(ImageIO.read(selectedFile), outputFile, (fileToSave, cardImage) -> {
                    try {
                        LOGGER.info("Saving file: " + String.valueOf(fileToSave.getAbsoluteFile()));
                        cardImage.saveToFile(fileToSave.getAbsolutePath());
                    }
                    catch (IOException e) {
                        LOGGER.log(Level.SEVERE, "Error saving file: " + String.valueOf(fileToSave.getAbsoluteFile()), e);
                    }
                }, String::valueOf, String::valueOf);
                JOptionPane.showMessageDialog(this, "Finalized atlas conversion.");
            }
            catch (IOException e) {
                LOGGER.log(Level.SEVERE, "Error loading image: " + String.valueOf(selectedFile.getAbsoluteFile()), e);
                JOptionPane.showMessageDialog(this, "Error loading image: " + selectedFile.getName());
            }
        }
    }

    private void fixCardColors() {
        JFileChooser fileChooser = this.getFileChooser();
        int returnValue = fileChooser.showOpenDialog(this);
        if (returnValue == 0) {
            File selectedFile = fileChooser.getSelectedFile();
            PREFERENCES.put(LAST_DIRECTORY, selectedFile.getParent());
            try {
                BufferedImage loadedImage = ImageIO.read(selectedFile);
                BufferedImage convertedImage = new BufferedImage(loadedImage.getWidth(), loadedImage.getHeight(), 2);
                try {
                    for (int x = 0; x < loadedImage.getWidth(); ++x) {
                        for (int y = 0; y < loadedImage.getHeight(); ++y) {
                            int oldArgb = loadedImage.getRGB(x, y);
                            int newArgb = CardEditor.fixColor(oldArgb);
                            convertedImage.setRGB(x, y, newArgb);
                        }
                    }
                    ImageIO.write((RenderedImage)convertedImage, "png", selectedFile);
                    JOptionPane.showMessageDialog(this, "Finished fixing card color.");
                }
                catch (IOException e) {
                    LOGGER.log(Level.SEVERE, "Error saving image: " + String.valueOf(selectedFile.getAbsoluteFile()), e);
                    JOptionPane.showMessageDialog(this, "Error saving image.");
                }
            }
            catch (IOException e) {
                LOGGER.log(Level.SEVERE, "Error loading image: " + String.valueOf(selectedFile.getAbsoluteFile()), e);
                JOptionPane.showMessageDialog(this, "Error loading image.");
            }
        }
    }

    private void generatePNGPalette() {
        JFileChooser fileChooser = this.getFileChooser();
        fileChooser.setDialogTitle("Specify a file to save");
        int userSelection = fileChooser.showSaveDialog(this);
        if (userSelection == 0) {
            File selectedFile = fileChooser.getSelectedFile();
            PREFERENCES.put(LAST_DIRECTORY, selectedFile.getParent());
            if (!selectedFile.getName().endsWith(".png")) {
                selectedFile = new File(selectedFile.getAbsolutePath() + ".png");
            }
            BufferedImage paletteImage = new BufferedImage(8, 8, 2);
            try {
                for (int y = 0; y < paletteImage.getWidth(); ++y) {
                    for (int x = 0; x < paletteImage.getHeight(); ++x) {
                        paletteImage.setRGB(x, y, -16777216 + CardImage.COLOR_PALETTE[x + y * 8]);
                    }
                }
                ImageIO.write((RenderedImage)paletteImage, "png", selectedFile);
                JOptionPane.showMessageDialog(this, "Exported palette image.");
            }
            catch (IOException e) {
                LOGGER.log(Level.SEVERE, "Error saving image: " + String.valueOf(selectedFile.getAbsoluteFile()), e);
                JOptionPane.showMessageDialog(this, "Error saving image.");
            }
        }
    }

    private void generateGPLPalette() {
        JFileChooser fileChooser = this.getFileChooser();
        fileChooser.setDialogTitle("Specify a file to save");
        int userSelection = fileChooser.showSaveDialog(this);
        if (userSelection == 0) {
            File selectedFile = fileChooser.getSelectedFile();
            PREFERENCES.put(LAST_DIRECTORY, selectedFile.getParent());
            if (!selectedFile.getName().endsWith(".gpl")) {
                selectedFile = new File(selectedFile.getAbsolutePath() + ".gpl");
            }
            try (BufferedWriter writer = new BufferedWriter(new FileWriter(selectedFile));){
                writer.write("GIMP Palette\n");
                writer.write("#\n");
                writer.write("# --------------------------------------------------------\n");
                writer.write("#      This is the Charta Minecraft Mod Card Palette.     \n");
                writer.write("#                    Yeah I know its bad.                 \n");
                writer.write("# --------------------------------------------------------\n");
                writer.write("#\n");
                for (int i = 0; i < CardImage.COLOR_PALETTE.length; ++i) {
                    int color = CardImage.COLOR_PALETTE[i];
                    int red = color >> 16 & 0xFF;
                    int green = color >> 8 & 0xFF;
                    int blue = color & 0xFF;
                    writer.write(String.format("%3d %3d %3d\t%s\n", red, green, blue, "Color " + (i + 1)));
                }
                JOptionPane.showMessageDialog(this, "Exported palette file.");
            }
            catch (IOException e) {
                LOGGER.log(Level.SEVERE, "Error saving palette: " + String.valueOf(selectedFile.getAbsoluteFile()), e);
                JOptionPane.showMessageDialog(this, "Error saving palette.");
            }
        }
    }

    private static int fixColor(int argb) {
        int oldRgb = argb & 0xFFFFFF;
        int oldAlpha = argb >> 24 & 0xFF;
        int colorIndex = CardImage.findClosestColorIndex(oldRgb);
        int alphaIndex = CardImage.findClosestAlphaIndex(oldAlpha);
        int newRgb = CardImage.COLOR_PALETTE[colorIndex];
        int newAlpha = CardImage.ALPHA_PALETTE[alphaIndex];
        return newAlpha << 24 | newRgb & 0xFFFFFF;
    }

    private void saveCardImage() {
        if (this.currentImage == null) {
            JOptionPane.showMessageDialog(this, "No image to save.");
            return;
        }
        JFileChooser fileChooser = this.getFileChooser();
        fileChooser.setDialogTitle("Specify a file to save");
        int userSelection = fileChooser.showSaveDialog(this);
        if (userSelection == 0) {
            File selectedFile = fileChooser.getSelectedFile();
            PREFERENCES.put(LAST_DIRECTORY, selectedFile.getParent());
            if (!selectedFile.getName().endsWith(".mccard")) {
                selectedFile = new File(selectedFile.getAbsolutePath() + ".mccard");
            }
            try {
                this.currentImage.saveToFile(selectedFile.getAbsolutePath());
                JOptionPane.showMessageDialog(this, "Image saved successfully.");
            }
            catch (IOException e) {
                LOGGER.log(Level.SEVERE, "Error saving image: " + String.valueOf(selectedFile.getAbsoluteFile()), e);
                JOptionPane.showMessageDialog(this, "Error saving image.");
            }
        }
    }

    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> {
            try {
                UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
            }
            catch (Exception e) {
                LOGGER.log(Level.SEVERE, "Error setting cross platform look and feel: ", e);
            }
            CardEditor editor = new CardEditor();
            editor.setUndecorated(true);
            editor.setBackground(new Color(0, 0, 0, 1));
            editor.getRootPane().setWindowDecorationStyle(1);
            editor.setVisible(true);
        });
    }
}

